#########################################################################
#	file: 			media_watchfolder_import.t							#
#	description:	Imports files from a watchfolder on disk into		#
#					farmerswife, creates/updates Media linked to a		#
#					Project and moves the files into the farmerswife	#
#					folder structure + more goodies.					#
#	created:		2014-06-12											#
#	created by:		Linus Nyberg										#
#	(C) Copyright 2014 Farmers WIFE S.L.								#
#########################################################################

########################## BEGIN SETUP ##################################### 
set ::WATCHFOLDER_LOG_FILENAME "media_watchfolder_importer_log.txt"
set ::WATCHFOLDER_LOCATION [file join $::FILES_HOME files media_watchfolder] ;# this is the path to the actual watchfolder on disk
set ::WATCHFOLDER_LIBNUM 0 ;# the index (should be in the range 0-11) of the Media Library to import into; 0 is the first Media Library.
set ::WATCHFOLDER_DATABASE_FILENAME "media_watchfolder_import_database.dat" ;# this is a file it uses to track which media it creates for which projects.
set ::ERROR_EMAILS_SEND_TO [list] ;# list of email addresses
########################## END OF SETUP ##################################### 

##### DON'T MODIFY ANYTHING BEYOND THIS POINT IF YOU'RE NOT REALLY SURE YOU KNOW WHAT YOU'RE DOING

set ::LOGF [open [file join $::SYSTEM_PATH $::WATCHFOLDER_LOG_FILENAME] a]

proc log {s {also_forward_to_wife 0}} {
	set timestamp [clock format [clock seconds] -format "%Y%m%d:%H%M%S"]
	if {$::LOGF != ""} {
		puts $::LOGF "$timestamp: $s"
	}
	if {$also_forward_to_wife} {
		wife::send log "$s"
	}
}

lassign [wife::send get_var FILES_HOME] ok ::FILES_HOME error_string
if {!$ok} {
	log "$error_string"
}

#log "Watchfolder Importer Starting..."

set ::WATCHFOLDER_DATABASE_LOADED 0
set ::WATCHFOLDER_DATABASE [list] ;# list of (prj_number lib_id)

proc watchfolder:load_database {} {
	if {$::WATCHFOLDER_DATABASE_LOADED} {
		return
	}
	if {![lindex [set ret [wife::send server:open_text_file2 $::WATCHFOLDER_DATABASE_FILENAME]] 0]} {
		# no such file - use empty database
		set ::WATCHFOLDER_DATABASE [list]
	} else {
		set ::WATCHFOLDER_DATABASE [string trim [lindex $ret 1]]
	}
	set ::WATCHFOLDER_DATABASE_LOADED 1
}

proc watchfolder:save_database {} {
	if {!$::WATCHFOLDER_DATABASE_LOADED} {
		return
	}
	if {![lindex [set ret [wife::send server:write_text_file2 $::WATCHFOLDER_DATABASE_FILENAME $::WATCHFOLDER_DATABASE]] 0]} {
		error "Error:\n[lindex $ret 2]"
	}
	return
}

proc watchfolder:get_master_id_for_project_number {prj_number} {
	watchfolder:load_database
	if {[set i [lsearch -exact -index 0 $::WATCHFOLDER_DATABASE $prj_number]] != -1} {
		return [lindex $::WATCHFOLDER_DATABASE $i 1]
	}
	
	# not found
	return ""
}

proc watchfolder:set_master_id_for_project_number {prj_number lib_id} {
	watchfolder:load_database
	if {[set i [lsearch -exact -index 0 $::WATCHFOLDER_DATABASE $prj_number]] != -1} {
		lset ::WATCHFOLDER_DATABASE $i 1 $lib_id
	} else {
		lappend ::WATCHFOLDER_DATABASE [list $prj_number $lib_id]
	}
	watchfolder:save_database
}

proc watchfolder:create_lib {project paths} {
	# build the new media:
	set owner [lindex $project 2]
	set lib_num $::WATCHFOLDER_LIBNUM
	set division_id [lindex $project 53]
	lassign [wife::send lib:build_new $owner $lib_num $division_id] ok lib error_string
	if {!$ok} {
		log "$error_string"
		return ""
	}
	
	lassign [wife::send generic:click] ok now error_string
	if {!$ok} {
		log "$error_string"
		return ""
	}

	# prj_id
	lset lib 17 [lindex $project 0]
	
	# use name from project
	lset lib 4 [lindex $project 4]
	
	# client
	lset lib 9 [lindex $project 6 3]
		
	# client_id (textual)
	lset lib 16 [lindex $project 6 4]
		
	# comp_id
	lset lib 18 [lindex $project 24]
	
	# history
	set his ""
	lappend his [list $now $owner "Media Was Automatically Created From Watchfolder"]
	lset lib 15 $his
	
	set lib [watchfolder:add_entries_to_lib $lib_num $lib $paths]

	log "Creating lib: $lib"

	# save it!
	lassign [wife::send lib:new $lib $lib_num] ok saved error_string
	if {!$ok} {
		log "$error_string"
		return ""
	}
	if {$saved != 1} {
		log "Failed Creating Media (Unknown Reason)"
		return ""
	}
	
	return $lib
}

proc watchfolder:update_existing_lib {project lib_num lib paths} {
	lassign [wife::send lib:lock_who [lindex $lib 0]] ok locked_by error_string
	if {!$ok} {
		log "$error_string"
		return ""
	}
	if {$locked_by != 0} {
		# it's locked
		log "Watchfolder import: Postponing import to Media for Project [lindex $project 5] because the Media is locked" 1
		return ""
	}
	
	set lib [watchfolder:add_entries_to_lib $lib_num $lib $paths]

	# save it!
	lassign [wife::send lib:modify $lib $lib_num] ok saved error_string
	if {!$ok} {
		log "$error_string"
		return ""
	}
	if {$saved != 1} {
		log "Failed Updating Media (Unknown Reason)"
		return ""
	}
	
	return $lib
}

proc watchfolder:add_entries_to_lib {lib_num lib paths} {
	set lib_entries [lindex $lib 14]
	foreach path $paths {
		set tail [file tail $path]
		set rootname [file rootname $tail]
		set prj_number [string range $rootname 0 [string first "_" $rootname]-1]
		set entry_title [string range $rootname [string first "_" $rootname]+1 end]
		if {[lsearch -exact -index 0 $lib_entries $entry_title] != -1} {
			# the entry already exists
			continue
		}

		log "   - Watchfolder importing file with title: $entry_title"

		# build the entry:
		lassign [wife::send lib:entry:build_new $lib_num $lib_entries] ok lib_entry error_string
		if {!$ok} {
			log "$error_string"
			continue
		}
		lset lib_entry 0 $entry_title
		lappend lib_entries $lib_entry
	}
	
	lset lib 14 $lib_entries
	
	return $lib
}

proc watchfolder:fail_import {paths reason} {
	# move the file to a sub folder called "failed"
	#----------------------------------------------
	foreach path $paths {
		set dest_folder [file join [file dirname $path] "failed"]
		if {![file exists $dest_folder]} {
			if {[catch {
				file mkdir $dest_folder
			} R1] == 1} {
				log "Watchfolder import ERROR: $R1" 1
			}
		}
		set dest_path [file join $dest_folder [file tail $path]]
		if {[catch {
			file rename -force -- $path $dest_path
		} R1] == 1} {
			log "Watchfolder import ERROR: $R1" 1
		}
	}

	# notify people about the failure
	#--------------------------------
	if {[llength $::ERROR_EMAILS_SEND_TO] > 0} {
		# let farmerswife send an email notifying about the errors
		set from "" ;# this means use admin email
		set to [join $::ERROR_EMAILS_SEND_TO ";"]
		set email_subject "Errors while importing files through watchfolder into farmerswife"
		set email_body "The following files could not be imported ($reason):\n"
		foreach path $paths {
			append email_body "- $path\n"
		}
		append email_body "\nThe files can be found in the sub folder called \"failed\"."
				
		if {![lindex [set ret [wife::send mail:send $from $to $email_subject $email_body]] 0]} {
			error "Error:\n[lindex $ret 2]"
		}
	}

	return
}

proc watchfolder:import {} {
	set lib_num $::WATCHFOLDER_LIBNUM

	# make sure the watchfolder exists:
	if {![file exists $::WATCHFOLDER_LOCATION]} {
		if {[catch {
			file mkdir $::WATCHFOLDER_LOCATION
		} R1] == 1} {
			log "Watchfolder import ERROR: $R1" 1
			return
		}
	}

	set to_import [list] ;# list of (project_number paths)
	foreach path [glob -nocomplain -types [list f] -directory $::WATCHFOLDER_LOCATION *.*] {
		set tail [file tail $path]
		set rootname [file rootname $tail]
		set prj_number [string range $rootname 0 [string first "_" $rootname]-1]
		set entry_title [string range $rootname [string first "_" $rootname]+1 end]
		if {$prj_number == "" || $entry_title == ""} {
			watchfolder:fail_import [list $path] "Empty project number"
			continue
		}
		if {[set i [lsearch -exact -index 0 $to_import $prj_number]] != -1} {
			set paths [lindex $to_import $i 1]
			lappend paths $path
			set paths [lsort -unique -dictionary $paths]
			lset to_import $i 1 $paths
		} else {
			lappend to_import [list $prj_number [list $path]]
		}
	}
	
	foreach elem $to_import {
		lassign $elem prj_number paths
		
		log "Watchfolder importing [llength $paths] files to project number $prj_number"

		# find the project:
		lassign [wife::send project:get_list_for_number $prj_number] ok project error_string
		if {!$ok} {
			log "$error_string"
			continue
		}
		
		if {$project == ""} {
			log "Watchfolder import ERROR: Could not find any project with number: $prj_number" 1
			watchfolder:fail_import $paths "Could not find any project with number: $prj_number"
			continue
		}
		
		# if the project doesn't have any name yet, skip it for now (someone might be create the project this very instant):
		if {[lindex $project 4] == ""} {
			log "Watchfolder import WARNING: Skipping import to project with number $prj_number because it doesn't have any name" 1
			watchfolder:fail_import $paths "Skipping import to project with number $prj_number because it doesn't have any name"
			continue
		}
		
		# create or update the master media for that project:
		set lib_id [watchfolder:get_master_id_for_project_number $prj_number]
		set lib ""
		if {$lib_id != ""} {
			lassign [wife::send lib:get_list $lib_id $lib_num] ok lib error_string
			if {!$ok} {
				log "$error_string"
				continue
			}
		}
		if {$lib == ""} {
			log "Watchfolder creating new Media"
			set lib [watchfolder:create_lib $project $paths]
			set lib_id [lindex $lib 0]
		} else {
			log "Watchfolder adding to existing Media"
			set lib [watchfolder:update_existing_lib $project $lib_num $lib $paths]
		}
		
		if {$lib == ""} {
			continue
		}
		
		# register it for future imports:
		watchfolder:set_master_id_for_project_number $prj_number [lindex $lib 0]
		
		# move the files into the entry folders:
		set imported_elems [list] ;# list of (dest_path lib_entry_id)
		foreach path $paths {
			set tail [file tail $path]
			set rootname [file rootname $tail]
			set prj_number [string range $rootname 0 [string first "_" $rootname]-1]
			set entry_title [string range $rootname [string first "_" $rootname]+1 end]

			set entry_i [lsearch -index 0 [lindex $lib 14] $entry_title]
			if {$entry_i == -1} {
				log "Watchfolder import ERROR: Could not find entry for $entry_title" 1
			}
			
			# build the entry:
			lassign [wife::send lib:get_entry_full_path [lindex $lib 0] $entry_i] ok entry_folder_path error_string
			if {!$ok} {
				log "$error_string"
				continue
			}
			
			log "   - Watchfolder moving file [file tail $path] into entry folder: $entry_folder_path"
			
			if {![file exists $entry_folder_path]} {
				if {[catch {
					file mkdir $entry_folder_path
				} R1] == 1} {
					log "Watchfolder import ERROR: $R1" 1
					continue
				}
			}
			if {[file exists $entry_folder_path]} {
				# set dest_path [file join $entry_folder_path [file tail $path]]
				# => Project Number gets stripped away
				set new_filename [string range $tail [string first "_" $tail]+1 end]
				set dest_path [file join $entry_folder_path $new_filename]
				if {[file exists $dest_path]} {
					log "Watchfolder import: Overwriting/updating existing file: $dest_path"
				}
				if {[catch {
					file rename -force -- $path $dest_path
					lappend imported_elems [list $dest_path [lindex $lib 14 $entry_i 10]]
				} R1] == 1} {
					log "Watchfolder import ERROR: $R1" 1
					continue
				}
			}
			
		}
		
		if {$::PLAY_OK} {
			# share the files on play
			#------------------------
			foreach imported_elem $imported_elems {
				lassign $imported_elem dest_path lib_entry_id

				# figure out the share creator:
				set project_owner_id [lindex $project 2 0]
				if {$project_owner_id != ""} {
					lassign [wife::send object:get_list "user" $project_owner_id] ok project_owner error_string
					if {!$ok} {
						log "Watchfolder import ERROR (while sharing on Play): $error_string" 1
						continue
					}
					set share_creator_email [lindex $project_owner 4]
					set share_creator_name "[lindex $project_owner 2] [lindex $project_owner 1]"
				}

				# first check if there are existing shares with this file, and if so update them:
				#--------------------------------------------------------------------------------
				set modification_note ""
				set updated_by_user_email $share_creator_email
				set updated_by_user_name $share_creator_name
				lassign [wife::send play:server:refesh_original_in_existing_shares_with_file_fw_server_pov $dest_path $modification_note $updated_by_user_email $updated_by_user_name] ok had_existing_shares error_string
				if {!$ok} {
					log "ERROR (Play Auto Sharing): $error_string" 1
					continue
				}
				if {$had_existing_shares} {
					log "Watchfolder updated existing Play Shares with file: $dest_path"
					# existing shares are updated - nothing more to do
					continue
				}
				
				# ok, there was no existing share - create a new one:
				#----------------------------------------------------
				log "Watchfolder sharing $dest_path on Play"

				lassign [wife::send play:server:auto_create_share_for_media [lindex $lib 0] "f" $dest_path $share_creator_email $share_creator_name -lib_entry_id $lib_entry_id] ok shared_files error_string
				if {!$ok} {
					log "ERROR (Play Auto Sharing): $error_string" 1
					continue
				}
				
				log "Watchfolder import created a share on Play with these files:"
				foreach shared_file $shared_files {
					log "   - [file tail $shared_file]"
				}
			}
		}

	}
}

watchfolder:import

if {$::LOGF != ""} {
	close $::LOGF
}
